<?php


namespace Mnv\Modules\User;

use Mnv\Core\Auth\Errors\MissingCallbackError;
use Mnv\Core\Config;
use Mnv\Core\Locale\I18N;
use Mnv\Core\Validations\ValidatePhone;
use Mnv\Models\Users\UserTypes;
use Mnv\Modules\SmsGateway;

use Mnv\Core\Auth\Errors\AuthError;
use Mnv\Core\Auth\Exceptions\UnknownIdException;
use Mnv\Core\Auth\Errors\UserPhoneRequiredError;
use Mnv\Core\Auth\Exceptions\NotLoggedInException;
use Mnv\Core\Auth\Exceptions\PhoneNotPaidException;
use Mnv\Core\Auth\Exceptions\EmailNotPaidException;
use Mnv\Core\Auth\Exceptions\InvalidPhoneException;
use Mnv\Core\Auth\Exceptions\TokenExpiredException;
use Mnv\Core\Auth\Exceptions\InvalidEmailException;
use Mnv\Core\Auth\Exceptions\ResetDisabledException;
use Mnv\Core\Auth\Exceptions\InvalidPasswordException;
use Mnv\Core\Auth\Exceptions\TooManyRequestsException;
use Mnv\Core\Auth\Exceptions\PhoneNotVerifiedException;
use Mnv\Core\Auth\Exceptions\AttemptCancelledException;
use Mnv\Core\Auth\Exceptions\ValidatePasswordException;
use Mnv\Core\Auth\Exceptions\UserAlreadyExistsException;
use Mnv\Core\Auth\Exceptions\InvalidSelectorTokenPairException;

use Mnv\Core\Database\Throwable\IntegrityConstraintViolationException;

class Individual extends User implements UserInterface
{
    /** @var SmsGateway */
    protected SmsGateway $sms;

    public $action;
    public array $user = [];
    public $uploadFile;
    public $address = [];

    /**
     * Phone constructor.
     */
    public function __construct()
    {
        parent::__construct();
        $this->sms = new SmsGateway(Config::getValue('sms_url'),Config::getValue('sms_login'), Config::getValue('sms_password'));
    }


    public function run(): void
    {
        switch ($this->action) {
            /** Проверка на существование номера телефона */
            case 'authCheck':
                $this->check();
                break;
            /** Подтверждение регистрации и авторизации через смс */
            case 'confirm':
                $this->confirmSms();
                break;
            case 'changeUser':
                $this->user['address'] = null;
                if (!empty($this->address)) {
                    $this->user['address'] = json_encode($this->address, JSON_UNESCAPED_UNICODE);
                }
                $this->changeUser($this->user);
                break;
            case 'changeContact':
                $this->changePhone();
                break;
            case 'changeConfirmContact':
                $this->changeConfirmContact();
                break;
            case 'repeatCode':
                $this->repeatCode();
                break;
//            case 'recoveryPhone':
//                $this->recoveryPhone();
//                break;
//            case 'recoveryConfirm':
//                $this->recoveryConfirmPhone();
//                break;
            default:
                $this->response =  array('status'=> 403, 'color'=> 'red', 'message'=> 'Error: method not found');
                break;
        }
    }
    /**
     * Проверка на существование номера телефона
     */
    public function check(): void
    {
        global $userAuth;

        if (isset($this->user['phone']) && !empty($this->user['phone'])) {

            try {
                $this->user['phone'] = ValidatePhone::fromString($this->user['phone']);
                try {
                    $this->user = $userAuth->getCheckUserPhoneNumber($this->user['phone']);
                    /** авторизоваться */
                    if ($this->user['verified'] == 1) {
                        $this->auth();
                    }
                    /** подтведить номер телефона */
                    else {
                        try {
                            $userAuth->createConfirmationRequestPhone($this->user['userId'], $this->user['phone'], 60, function ($phone, $confirmCode) {
                                $text = I18N::locale("Ваш код подтверждения на сайте : $confirmCode", "Saytida tasdiqlash kodi: $confirmCode", "Your confirmation code on the website : $confirmCode");
                                $this->response = $this->sendSms($this->user['phone'], $this->user['firstName'], $text, 'confirm', true);
                            });
                        } catch (MissingCallbackError $e) {
                            $this->response =  array('status'=> 403, 'color'=> 'red', 'message'=> $e->getMessage());
                        }
                    }
                } /** отправляем на регистрацию */
                catch (InvalidPhoneException $e) {
                    $this->registration();
                }

            } catch (\Mnv\Core\Validations\Exceptions\InvalidPhoneException $e) {
                $this->response =  array('status'=> 403, 'color'=> 'red', 'message'=> I18N::locale(
                    "Не верный формат номера телефона", "Noto'g'ri telefon raqami formati", "Invalid phone number format")
                );
            }
        } else {
            $this->response =  array('status'=> 403, 'color'=> 'red', 'message'=> I18N::locale(
                "Поле номера телефона не должно быть пустым!", "Telefon raqami maydoni bo'sh bo'lmasligi kerak!", "The phone number field should not be empty!"
            ));
        }
    }

    /** Авторизация */
    public function auth(): void
    {
        global $userAuth;

        $rememberDuration = (isset($this->user['remember']) && $this->user['remember'] == 1) ? (int) (60 * 60 * 24 * 365.25) : null;
        try {
            /** подтверждение через смс шлюз */
            $callback = function ($phone, $tokenSms) {
                $text = I18N::locale("Ваш активационный код на сайте: $tokenSms", "Ваш активационный код на сайте: $tokenSms", "Ваш активационный код на сайте: $tokenSms");
                $this->response = $this->sendSms($phone, $this->user['firstName'], $text, 'auth', true);
            };

            $userAuth->loginWithPhone($this->user['phone'], (int)Config::getValue('twoFactorAuth'), $rememberDuration, $callback);

        }
        catch (InvalidPhoneException $e) {
            $this->response = array('status' => 403, 'color'=> 'red', 'message' => I18N::locale(
                "Пользователь с таким номером телефона не зарегистрирован", "Ushbu telefon raqamiga ega foydalanuvchi ro'yxatdan o'tmagan", "The user with this phone number is not registered"
            ));
        }
        catch (UnknownIdException $e) {
            $this->response = array('status' => 403, 'color'=> 'red', 'message' => I18N::locale(
                "Пользователь не найден", "Foydalanuvchi topilmadi", 'The user was not found'
            ));
        }
        catch (PhoneNotVerifiedException $e) {
            $this->response = array('status' => 403, 'color'=> 'red', 'message' => I18N::locale(
                "Номер телефона не был подтвержден", "Telefon raqami tasdiqlanmagan", "The phone number has not been confirmed"
            ));
        }
        catch (TooManyRequestsException $e) {
            $this->response =  array('status'=> 403, 'color'=> 'red', 'message'=> I18N::locale(
                "Слишком много запросов", "Juda ko'p so'rovlar", 'Too many requests'
            ));
        } catch (AttemptCancelledException $e) {
            $this->response =  array('status'=> 403, 'color'=> 'red', 'message'=> I18N::locale(
                "Попытка отменена", "Urinish bekor qilindi", "The attempt was canceled"
            ));
        } catch (IntegrityConstraintViolationException $e) {
            $this->response = array('status' => 403, 'color'=> 'red', 'message' => I18N::locale(
                "Ваш ip заблокирован из-за нарушений авторизации", "Sizning ip avtorizatsiya buzilishi tufayli bloklangan", "Your IP has been blocked due to authorization violations"
            ));
        } catch (AuthError $e) {
            $this->response =  array('status'=> 403, 'color'=> 'red', 'message'=> $e->getMessage());
        }

    }

    /**
     * Регистрация по номеру телефона
     */
    public function registration(): void
    {
        global $userAuth;

        try {
            $userAuth->registerWithUniqueUsername($this->user, function ($phone, $confirmCode) {
                /** подтверждение через смс шлюз */
                $text = I18N::locale("Ваш код подтверждения на сайте: $confirmCode", "Saytida tasdiqlash kodi: $confirmCode", "Your website confirmation code: $confirmCode");
                $this->response = $this->sendSms($phone, '', $text, 'registration', true);
            });
        }
        catch (UserAlreadyExistsException $e) {
            $this->response =  array('status'=> 403, 'color'=> 'red', 'message'=> I18N::locale(
                "Такой номер телефона уже зарегистрирован", "Bu telefon raqami allaqachon roʻyxatdan oʻtgan", "This phone number is already registered"
            ));
        }
        catch (TooManyRequestsException $e) {
            $this->response =  array('status'=> 403, 'color'=> 'red', 'message' => I18N::locale(
                "Слишком много запросов", "Juda ko'p so'rovlar", "Too many requests"
            ));
        }

    }

    /** Подтверждение регистрации и авторизации через смс */
    public function confirmSms(): void
    {
        global $userAuth;

        if (!empty($this->user['code'])) {
            try {
                $userAuth->confirmPhoneAndSignIn($this->user['code'], $this->user['fullName'] ?? null);
                $redirect = SITE_URL . '/user/';
//                print_r($this->user);
                if ($this->user['redirect']) {
                    $redirect = SITE_URL . '/cart/checkout/';
                }

                $this->response =  array('status'=> 200, 'color'=> 'green', 'redirect' => $redirect, 'message'=> I18N::locale(
                    "Добро пожаловать!", "Xush kelibsiz!", "Welcome!"
                ));

            } catch (InvalidSelectorTokenPairException $e) {
                $this->response =  array('status'=> 403, 'color'=> 'red', 'message'=> I18N::locale(
                    "Не верный код подтверждения!", "Tasdiq kodi noto'g'ri!", "Invalid confirmation code!"
                ));
            } catch (UserAlreadyExistsException $e) {
                $this->response =  array('status'=> 403, 'color'=> 'red', 'message'=> I18N::locale(
                    "Такой номер телефона уже зарегистрирован", "Bu telefon raqami allaqachon roʻyxatdan oʻtgan", "This phone number is already registered"
                ));
            } catch (TokenExpiredException $e) {
                $this->response =  array('status'=> 403, 'color'=> 'red', 'message'=> I18N::locale(
                    "Время истекло", "Vaqt tugadi", "Time is up"
                ));
            } catch (TooManyRequestsException $e) {
                $this->response =  array('status'=> 403, 'color'=> 'red', 'message'=> I18N::locale(
                    "Слишком много запросов", "Juda koʻp soʻrovlar", "Too many requests"
                ));
            } catch (AuthError $e) {
                $this->response =  array('status'=> 403, 'color'=> 'red', 'message'=> $e->getMessage());
            }
        } else {
            $this->response =  array('status'=> 403, 'color'=> 'red', 'message'=> I18N::locale(
                "Отправленные данные пусты!", "Yuborilgan ma'lumotlar bo'sh!", "Sent data is empty!"
            ));
        }
    }

    /**
     * Изменение номера телефона
     */
    public function changePhone(): void
    {
        global $userAuth;

        if (!empty($this->user)) {
            $this->user['phone'] = $this->sms->formatPhoneNumber($this->user['phone']);
            try {
                $userAuth->changeConfirmUserPhone($this->user['phone'], 60, function ($phone, $confirmCode) {
                    $phone = $this->sms->formatPhoneNumber($phone);
                    $text = I18N::locale(
                        "Ваш код подтверждения для изменения номера телефона на сайте: $confirmCode",
                        "Saytdagi telefon raqamingizni o'zgartirish uchun tasdiqlash kodingiz: $confirmCode",
                        "Your confirmation code for changing your phone number on the site: $confirmCode"
                    );
                    if ($this->sms->sendSms($phone, $text)) {
                        $phoneHide = substr_replace($phone,'****',4,13);

                        $this->response = array('status' => 200, 'sms' => true, 'confirm' => false, 'color'=> 'green', 'action' => 'changeConfirmContact',
                            'message' => I18N::locale(
                                "На Ваш номер $phoneHide телефона отправлен код подтверждения!",
                                "Tasdiqlash kodi $phoneHide telefon raqamingizga yuborildi!",
                                "A verification code has been sent to your $phoneHide phone number!"
                            ),
                            'button' => I18N::locale("Подтвердить", "Tasdiqlash", "Confirm"),
                        );

                    } else {
                        $this->response = array('status' => 403, 'color'=> 'red',
                            'message' => I18N::locale("Настройте правильно смс шлюз!", "Nastroyte to'g'ri sms shlyuz!", "Set up the right sms gateway!")
                        );
                    }
                });
            }  catch (NotLoggedInException $e) {
                $this->response =  array('status'=> 403, 'color'=> 'red',
                    'message'=> I18N::locale("Вы не авторизованы", "Sizga ruxsat berilmagan", "You are not authorized")
                );
            } catch (UserAlreadyExistsException $e) {
                $this->response =  ['status'=> 403, 'color'=> 'red',
                    'message'=> I18N::locale("Такой номер телефона уже зарегистрирован", "Bu telefon raqami allaqachon roʻyxatdan oʻtgan", "This phone number is already registered")
                ];
            } catch (TooManyRequestsException $e) {
                $this->response =  ['status'=> 403, 'color'=> 'red',
                    'message'=> I18N::locale("Слишком много запросов!", "Too many requests!", "Too many requests!")
                ];
            }
        } else {
            $this->response =  array('status'=> 403, 'color'=> 'red',
                'message'=> I18N::locale("Отправленные данные пусты!", "Yuborilgan ma'lumotlar bo'sh!", "Sent data is empty!")
            );
        }
    }

    /**
     * Подтверждение на изменение номера телефона
     */
    public function changeConfirmContact(): void
    {
        global $userAuth;

        if (!empty($this->user['code'])) {
            try {
                $phoneBeforeAndAfter = $userAuth->confirmPhone($this->user['code']);
                if ($phoneBeforeAndAfter[1] !== null) {
                    $this->response = array('status' => 200, 'color' => 'green', 'sms' => false, 'confirm' => true,
                        'message' => I18N::locale("Ваш номер телефона успешно изменен", "Telefon raqamingiz muvaffaqiyatli o'zgartirildi", "Your phone number has been successfully changed"),
                        'button' => I18N::locale("Изменить", "O'zgartirish", "Change"),
                    );
                } else {
                    $this->response = array('status' => 200, 'color' => 'green', 'sms' => false, 'confirm' => true,
                        'message' => I18N::locale("Ошибка изменения номера", "Raqamni o'zgartirish xatosi", "Number change error"),
                        'button' => I18N::locale("Ошибка", "Xato", "Error"),
                    );
                }

            } catch (InvalidSelectorTokenPairException $e) {
                $this->response =  array('status'=> 403, 'color'=> 'red',
                    'message'=> I18N::locale("Не верный код подтверждения!", "Tasdiq kodi noto'g'ri!", "Invalid confirmation code!")
                );
            } catch (UserAlreadyExistsException $e) {
                $this->response =  array('status'=> 403, 'color'=> 'red',
                    'message'=> I18N::locale("Такой номер телефона уже зарегистрирован", "Bu telefon raqami allaqachon roʻyxatdan oʻtgan", "This phone number is already registered")
                );
            }  catch (TokenExpiredException $e) {
                $this->response =  array('status'=> 403, 'color'=> 'red',
                    'message'=> I18N::locale("Время истекло", "Vaqt tugadi", "Time is up")
                );
            } catch (TooManyRequestsException $e) {
                $this->response =  array('status'=> 403, 'color'=> 'red',
                    'message'=> I18N::locale("Слишком много запросов", "Juda ko'p soʻrovlar", "Too many requests")
                );
            } catch (AuthError $e) {
                $this->response =  array('status'=> 403, 'color'=> 'red', 'message'=> $e->getMessage());
            }
        } else {
            $this->response =  array('status'=> 403, 'color'=> 'red',
                'message'=> I18N::locale("Отправленные данные пусты!", "Yuborilgan ma'lumotlar bo'sh!", "Sent data is empty!")
            );
        }

    }

    /** не получил код */
    private function repeatCode(): void
    {
        global $userAuth;

        if (!empty($this->user['phone'])) {
            $rememberDuration = (isset($this->user['remember']) && $this->user['remember'] == 1) ? (int) (60 * 60 * 24 * 365.25) : null;

            if ((int)Config::getValue('twoFactorAuth') == 1) {
                $callback = function ($phone, $tokenSms) {  /** подтверждение через смс шлюз */
                    $phone = $this->sms->formatPhoneNumber($phone);
                    if ($this->sms->sendSms($phone, I18N::locale("Ваш код подтверждения на сайте " . GLOBAL_URL . " : $tokenSms", "Saytdagi tasdiqlash kodingiz " . GLOBAL_URL . " : $tokenSms", "Your confirmation code on the website" . GLOBAL_URL . " : $tokenSms")))  {
                        $phoneHide = substr_replace($phone,'****',4,13);
                        $this->response = array('status' => 200, 'sms' => true, 'type' => 'success', 'phone' => $phone, 'message' => I18N::locale(
                            "На Ваш номер $phoneHide телефона отправлен код подтверждения!",
                            "$phoneHide telefon raqamingizga tasdiqlash kodi!",
                            "Confirmation code to your phone number $phoneHide!"
                        ));
                    } else {
                        $this->response = array('status' => 403, 'type'=> 'error', 'message' => I18N::locale("Настройте правильно смс шлюз!", "Nastroyte to'g'ri sms shlyuz!", "Set up the right sms gateway!"));
                    }
                };
            } else {
                $callback = null;
            }

            try {
                $userAuth->createRepeatPhoneCode($this->user['phone'], $rememberDuration, $callback);
            } catch (MissingCallbackError $e) {
                $this->response = array('status' => 403, 'type'=> 'error',
                    'message' => I18N::locale("Ошибка callback", "Error callback", 'Error callback')
                );
            } catch (InvalidPhoneException $e) {
                $this->response = array('status' => 403, 'type'=> 'error', 'message' => I18N::locale(
                    "Пользователь с таким номером телефона не зарегистрирован",
                    "Ushbu telefon raqamiga ega foydalanuvchi ro'yxatdan o'tmagan",
                    "The user with this phone number is not registered"
                ));
            }
        }

    }


    private function sendSms($phone, $firstName, $text, $method, $confirm = false): array
    {
        $phone = $this->sms->formatPhoneNumber($phone);

        if ($this->sms->sendSms($phone, $text))  {
            $phoneHide = substr_replace($phone,'****',4,13);
            $response = array('status' => 200, 'method' => $method, 'name' => $firstName, 'confirm' => $confirm, 'color'=> 'green', 'message' => I18N::locale(
                "На Ваш номер $phoneHide телефона отправлен код подтверждения!",
                "Telefoningizning $phoneHide raqamiga tasdiqlash kodi yuborildi!",
                "A confirmation code has been sent to your $phoneHide phone number!"
            ));
        } else {
            $response = array('status' => 403, 'color'=> 'red', 'message' => I18N::locale(
                "Настройте правильно смс шлюз!",
                "Nastroyte to'g'ri sms shlyuz!",
                "Set up the right sms gateway!"
            ));
        }

        return $response;

    }

}